#!/bin/bash
set -e
exec > >(tee -i build.log)
exec 2>&1

# Configuration
BASE_DIR="$(pwd)"
IMAGE_NAME="ubuntu-24.04.3-rpi3-minimal.img"
IMAGE_SIZE="2000M"
UBUNTU_BASE_URL="https://cdimage.ubuntu.com/ubuntu-base/releases/24.04.3/release/ubuntu-base-24.04.3-base-arm64.tar.gz"
UBUNTU_BASE_FILE="ubuntu-base-24.04.3-base-arm64.tar.gz"
UBUNTU_BASE_SHA256="7b2dced6dd56ad5e4a813fa25c8de307b655fdabc6ea9213175a92c48dabb048"
FIRMWARE_REPO="https://github.com/raspberrypi/firmware.git"
FIRMWARE_DIR="rpi-firmware"
MOUNT_DIR="/mnt/rpi_root"
BOOT_MOUNT_DIR="/mnt/rpi_boot"
WIFI_SSID="YOUR_SSID"
WIFI_PASSWORD="YOUR_PASSWORD"

# Fix Git Ownership Issue for WSL
git config --global --add safe.directory "$BASE_DIR/$FIRMWARE_DIR" || true

# 1. Download Resources
echo "--- Step 1: Checking Resources ---"
if [ ! -f "$BASE_DIR/$UBUNTU_BASE_FILE" ]; then
    echo "Downloading Ubuntu Base..."
    wget -q "$UBUNTU_BASE_URL" -O "$BASE_DIR/$UBUNTU_BASE_FILE"
fi

echo "Verifying checksum..."
echo "$UBUNTU_BASE_SHA256  $BASE_DIR/$UBUNTU_BASE_FILE" | sha256sum -c - || {
    echo "Checksum failed! Deleting corrupted file..."
    rm "$BASE_DIR/$UBUNTU_BASE_FILE"
    exit 1
}

if [ ! -d "$BASE_DIR/$FIRMWARE_DIR" ]; then
    echo "Cloning Raspberry Pi Firmware (Depth 1)..."
    git clone --depth 1 "$FIRMWARE_REPO" "$BASE_DIR/$FIRMWARE_DIR"
else
    echo "Firmware directory exists. Skipping update..."
    # Use subshell to avoid changing main script CWD
    #(cd "$BASE_DIR/$FIRMWARE_DIR" && git pull)
fi

# 2. Create and Partition Image
echo "--- Step 2: Creating Image ---"
cd "$BASE_DIR" # Ensure we are back in base di
rm -f "$IMAGE_NAME"
truncate -s "$IMAGE_SIZE" "$IMAGE_NAME"

echo "Partitioning..."
# MBR: 256MB FAT32 Boot, Rest EXT4 Root
parted -s "$IMAGE_NAME" mklabel msdos
parted -s "$IMAGE_NAME" mkpart primary fat32 1MiB 257MiB
parted -s "$IMAGE_NAME" set 1 boot on
parted -s "$IMAGE_NAME" mkpart primary ext4 257MiB 100%

# 3. Loop Setup
echo "--- Step 3: Loop Setup ---"
LOOP_DEV=$(sudo losetup -fP --show "$IMAGE_NAME")
echo "Loop device: $LOOP_DEV"

cleanup() {
    if [ -n "$LOOP_DEV" ]; then
        echo "Cleaning up..."
        sudo umount "$BOOT_MOUNT_DIR" 2>/dev/null || true
        sudo umount "$MOUNT_DIR/dev/pts" 2>/dev/null || true
        sudo umount "$MOUNT_DIR/dev" 2>/dev/null || true
        sudo umount "$MOUNT_DIR/proc" 2>/dev/null || true
        sudo umount "$MOUNT_DIR/sys" 2>/dev/null || true
        sudo umount "$MOUNT_DIR" 2>/dev/null || true
        sudo losetup -d "$LOOP_DEV" 2>/dev/null || true
    fi
}
trap cleanup EXIT

# 4. Format
echo "Formatting Boot (FAT32)..."
sudo mkfs.vfat -n BOOT -F 32 "${LOOP_DEV}p1"
echo "Formatting Root (EXT4)..."
sudo mkfs.ext4 -L cloudimg-rootfs "${LOOP_DEV}p2"

# 5. Mount
echo "--- Step 4: Mounting ---"
sudo mkdir -p "$MOUNT_DIR" "$BOOT_MOUNT_DIR"
sudo mount "${LOOP_DEV}p2" "$MOUNT_DIR"
sudo mount "${LOOP_DEV}p1" "$BOOT_MOUNT_DIR"

# 6. Extract Rootfs
echo "--- Step 5: Extracting Rootfs ---"
# Use absolute path to avoid CWD confusion
if [ ! -f "$BASE_DIR/$UBUNTU_BASE_FILE" ]; then
    echo "Error: Ubuntu Base not found at $BASE_DIR/$UBUNTU_BASE_FILE"
    exit 1
fi
sudo tar -xpf "$BASE_DIR/$UBUNTU_BASE_FILE" -C "$MOUNT_DIR"

# 6.5 Bind Mounts for Chroot
echo "--- Step 5.5: Bind Mounting for Chroot ---"
sudo mount --bind /dev "$MOUNT_DIR/dev"
sudo mount --bind /dev/pts "$MOUNT_DIR/dev/pts"
sudo mount --bind /proc "$MOUNT_DIR/proc"
sudo mount --bind /sys "$MOUNT_DIR/sys"

# 7. Install Kernel & Firmware (Manual Method)
echo "--- Step 6: Installing Kernel & Firmware ---"
# Copy Boot Files
sudo cp -r "$BASE_DIR/$FIRMWARE_DIR/boot/"* "$BOOT_MOUNT_DIR/"

# Ensure kernel8.img exists (RPi 3B 64-bit)
if [ -f "$BOOT_MOUNT_DIR/kernel8.img" ]; then
    echo "kernel8.img found."
else
    echo "Warning: kernel8.img not found. Looking for alternatives..."
fi

# Copy Modules
echo "Copying Kernel Modules..."
sudo mkdir -p "$MOUNT_DIR/lib/modules"
sudo cp -r "$BASE_DIR/$FIRMWARE_DIR/modules/"* "$MOUNT_DIR/lib/modules/"

# 8. Configure Boot
echo "--- Step 7: Configuring Boot ---"

# cmdline.txt
# Changing root from LABEL to /dev/mmcblk0p2 to avoid need for initramfs
echo "console=serial0,115200 console=tty1 root=/dev/mmcblk0p2 rootfstype=ext4 fsck.repair=yes rootwait net.ifnames=0" | sudo tee "$BOOT_MOUNT_DIR/cmdline.txt"

# config.txt
cat <<EOF | sudo tee "$BOOT_MOUNT_DIR/config.txt"
# 64-bit Boot
arm_64bit=1
kernel=kernel8.img

# Watchdog (Auto-reboot if frozen for 15s)
dtparam=watchdog=on

# Display
dtoverlay=vc4-kms-v3d

# Interfaces
enable_uart=1
dtparam=i2c_arm=on
dtparam=spi=on

# Audio
dtparam=audio=on

# Camera
start_x=1
gpu_mem=128
EOF

# fstab
cat <<EOF | sudo tee "$MOUNT_DIR/etc/fstab"
LABEL=cloudimg-rootfs   /               ext4    defaults,noatime  0       1
LABEL=BOOT              /boot/firmware  vfat    defaults          0       2
tmpfs                   /tmp            tmpfs   defaults,noatime,nosuid,nodev 0 0
EOF

# 9. Network & Hostname
echo "ubuntu-rpi3" | sudo tee "$MOUNT_DIR/etc/hostname"
cat <<EOF | sudo tee "$MOUNT_DIR/etc/hosts"
127.0.0.1       localhost
127.0.1.1       ubuntu-rpi3
EOF

# Netplan (NetworkManager)
sudo mkdir -p "$MOUNT_DIR/etc/netplan"
# Use NetworkManager as it is more robust for WiFi on RPi than networkd
cat <<EOF | sudo tee "$MOUNT_DIR/etc/netplan/01-netcfg.yaml"
network:
  version: 2
  renderer: NetworkManage
  ethernets:
    eth0:
      dhcp4: true
  wifis:
    wlan0:
      dhcp4: true
      access-points:
        "$WIFI_SSID":
          password: "$WIFI_PASSWORD"
EOF

sudo chmod 600 "$MOUNT_DIR/etc/netplan/01-netcfg.yaml"

# 10. User Setup (Chroot)
echo "--- Step 8: User Setup (Chroot) ---"
sudo cp /usr/bin/qemu-aarch64-static "$MOUNT_DIR/usr/bin/"

# Prepare chroot script
cat <<EOF | sudo tee "$MOUNT_DIR/setup_user.sh"
#!/bin/bash
# Enable Apt Caching if available
if [ -n "\$APT_PROXY" ]; then
    echo "Acquire::http::Proxy \"\$APT_PROXY\";" > /etc/apt/apt.conf.d/00proxy
fi

# Fix DNS first
rm -f /etc/resolv.conf
echo "nameserver 8.8.8.8" > /etc/resolv.conf

# Fix Sources (Remove default noble sources to avoid duplicates/conflicts)
rm -f /etc/apt/sources.list.d/ubuntu.sources
echo "deb http://ports.ubuntu.com/ubuntu-ports noble main universe restricted multiverse" > /etc/apt/sources.list
echo "deb http://ports.ubuntu.com/ubuntu-ports noble-updates main universe restricted multiverse" >> /etc/apt/sources.list
echo "deb http://ports.ubuntu.com/ubuntu-ports noble-security main universe restricted multiverse" >> /etc/apt/sources.list

apt-get update

# Install Minimal Init System and Utils
# Added cloud-guest-utils for growpart
# Pre-configure users/groups to avoid tmpfiles.d errors
# We let the system pick GID/UID to avoid conflicts, but ensure they exist.
if ! getent group systemd-journal >/dev/null; then groupadd -r systemd-journal; fi
if ! getent group systemd-network >/dev/null; then groupadd -r systemd-network; fi
if ! getent group systemd-resolve >/dev/null; then groupadd -r systemd-resolve; fi
if ! getent group systemd-timesync >/dev/null; then groupadd -r systemd-timesync; fi

if ! getent passwd systemd-network >/dev/null; then 
    useradd --system --no-create-home --home-dir /run/systemd --shell /usr/sbin/nologin -g systemd-network systemd-network
fi
if ! getent passwd systemd-resolve >/dev/null; then 
    useradd --system --no-create-home --home-dir /run/systemd --shell /usr/sbin/nologin -g systemd-resolve systemd-resolve
fi
if ! getent passwd systemd-timesync >/dev/null; then 
    useradd --system --no-create-home --home-dir /run/systemd --shell /usr/sbin/nologin -g systemd-timesync systemd-timesync
fi

# Prevent services from starting during install
echo -e "#!/bin/sh\nexit 101" > /usr/sbin/policy-rc.d
chmod +x /usr/sbin/policy-rc.d

# Install 'file' command first as it is needed by NetworkManager postinst
DEBIAN_FRONTEND=noninteractive apt-get install -y file

DEBIAN_FRONTEND=noninteractive apt-get install -y \
    systemd-sysv \
    udev \
    kmod \
    sudo \
    net-tools \
    network-manager \
    wireless-tools \
    wpasupplicant \
    nano \
    openssh-server \
    iproute2 \
    iputils-ping \
    isc-dhcp-client \
    wget \
    locales \
    tzdata \
    cloud-guest-utils \
    rng-tools5 \
    zram-tools

# Enable Hardware RNG (speeds up boot/SSH)
echo "HRNGDEVICE=/dev/hwrng" >> /etc/default/rng-tools-debian

# Configure ZRAM (50% of RAM for swap)
echo "ALGO=lz4" >> /etc/default/zramswap
echo "PERCENT=50" >> /etc/default/zramswap

# Restore policy-rc.d
rm -f /usr/sbin/policy-rc.d

# Download RPi 3B WiFi & Bluetooth Firmware (BCM43430)
mkdir -p /lib/firmware/brcm
# WiFi - Using upstream Linux firmware (Cypress) for bin/clm_blob and RPi-Distro for config
wget -q "https://git.kernel.org/pub/scm/linux/kernel/git/firmware/linux-firmware.git/plain/cypress/cyfmac43430-sdio.bin" -O /lib/firmware/brcm/brcmfmac43430-sdio.bin
wget -q "https://git.kernel.org/pub/scm/linux/kernel/git/firmware/linux-firmware.git/plain/cypress/cyfmac43430-sdio.clm_blob" -O /lib/firmware/brcm/brcmfmac43430-sdio.clm_blob
wget -q "https://raw.githubusercontent.com/RPi-Distro/firmware-nonfree/bullseye/debian/config/brcm80211/brcm/brcmfmac43430-sdio.txt" -O /lib/firmware/brcm/brcmfmac43430-sdio.txt

# Create necessary symlink for RPi 3B
ln -sf brcmfmac43430-sdio.txt /lib/firmware/brcm/brcmfmac43430-sdio.raspberrypi,3-model-b.txt

# Bluetooth
wget -q "https://github.com/RPi-Distro/bluez-firmware/raw/master/broadcom/BCM43430A1.hcd" -O /lib/firmware/brcm/BCM43430A1.hcd
# Regulatory DB
wget -q "https://git.kernel.org/pub/scm/linux/kernel/git/sforshee/wireless-regdb.git/plain/regulatory.db" -O /lib/firmware/regulatory.db
wget -q "https://git.kernel.org/pub/scm/linux/kernel/git/sforshee/wireless-regdb.git/plain/regulatory.db.p7s" -O /lib/firmware/regulatory.db.p7s

# Generate Locale
locale-gen en_US.UTF-8
update-locale LANG=en_US.UTF-8

# Create Hardware Groups
groupadd -f -r gpio
groupadd -f -r i2c
groupadd -f -r spi
groupadd -f -r video

# Create Use
useradd -m -s /bin/bash ubuntu
echo "ubuntu:changeme" | chpasswd
usermod -aG sudo,video,gpio,i2c,spi ubuntu

# Enable SSH
systemctl enable ssh

# Remove unique SSH keys so they regenerate on first boot
rm -f /etc/ssh/ssh_host_*

# --- Auto-Expand Filesystem Setup ---
echo "Setting up auto-expansion script..."
cat <<'SCRIPT' > /usr/local/bin/resize-rootfs.sh
#!/bin/bash
set -e
echo "Resizing root partition..."
# Grow partition 2 on mmcblk0
# Note: growpart requires a space between device and partition numbe
growpart /dev/mmcblk0 2 || echo "growpart failed or already grown"

echo "Resizing filesystem..."
resize2fs /dev/mmcblk0p2 || echo "resize2fs failed"

echo "Disabling resize service..."
systemctl disable resize-rootfs
rm -f /etc/systemd/system/resize-rootfs.service
rm -f /usr/local/bin/resize-rootfs.sh
systemctl daemon-reload
echo "Resize complete."
SCRIPT

chmod +x /usr/local/bin/resize-rootfs.sh

cat <<'SERVICE' > /etc/systemd/system/resize-rootfs.service
[Unit]
Description=Resize Root Filesystem
After=local-fs.target
Wants=local-fs.target

[Service]
Type=oneshot
ExecStart=/usr/local/bin/resize-rootfs.sh
RemainAfterExit=yes

[Install]
WantedBy=multi-user.target
SERVICE

systemctl enable resize-rootfs
# ------------------------------------

EOF

sudo chmod +x "$MOUNT_DIR/setup_user.sh"
sudo chroot "$MOUNT_DIR" /bin/bash /setup_user.sh

# Cleanup Chroot Script
sudo rm "$MOUNT_DIR/setup_user.sh"
sudo rm "$MOUNT_DIR/usr/bin/qemu-aarch64-static"
sudo rm -f "$MOUNT_DIR/etc/apt/apt.conf.d/00proxy"

# 11. Final Cleanup
echo "--- Step 9: Cleanup ---"
echo "Disk Usage before cleanup:"
df -h "$MOUNT_DIR"
# Trap will handle unmounting, but doing it explicitly here is good practice
sudo umount "$BOOT_MOUNT_DIR"
sudo umount "$MOUNT_DIR/dev/pts"
sudo umount "$MOUNT_DIR/dev"
sudo umount "$MOUNT_DIR/proc"
sudo umount "$MOUNT_DIR/sys"
sudo umount "$MOUNT_DIR"
sudo losetup -d "$LOOP_DEV"
LOOP_DEV=""

echo "Compressing..."
# Use parallel xz if available
if command -v pixz >/dev/null 2>&1; then
    echo "Using pixz for parallel compression..."
    pixz "$IMAGE_NAME"
elif command -v xz >/dev/null 2>&1; then
    echo "Using xz (single-threaded) compression..."
    xz -f -v -T0 "$IMAGE_NAME"
else
    echo "xz not found, falling back to gzip..."
    gzip -9 "$IMAGE_NAME"
fi

echo "Done! Image is $IMAGE_NAME.xz"
